module Bounded
(	Bound,		-- abstract

	newBound,	{- :: Int -> IO (Bound a)
				Creates a FIFO buffer with the specified maximum size,
				ie a "bounded buffer". -}
	writeBound,	{- :: Bound a -> a -> IO ()
				Write a single value to the buffer.
				NOTE: Will block if it is full. -}
	readBound,	{- :: Bound a -> IO a
				Read a single value from the buffer.
				NOTE: Will block if it is empty. -}

	isEmptyBound,	{- :: Bound a -> IO Bool
				Check if buffer is empty at this specific instant in time. -}
	isFullBound,	{- :: Bound a -> IO Bool
				Check if buffer is full at this specific instant in time. -}

	snapBound,	{- :: Bound a -> IO [a] 
				Generate a list representing a snapshot view of the 
				buffer contents, the buffer is unchanged. -}
--	flushBound,	-- :: Bound a -> IO [a]
--	addBound	-- :: Bound a -> [a] -> IO ()

) where

import MutSig
import Distributed

type Bound a = (Mutex [a], Signal, Signal, Int) 

instance Immobile (Bound a) where 
  owningPE (m,_,_,_) = owningPE m

newBound :: Int -> IO (Bound a)
newBound l = do
	m <-newMutex []
	ne <-newSignal
	nf <-newSignal
	return (m,ne,nf,l)

empty :: [a] -> Bool
empty [] = True
empty _  = False

full :: Int -> [a] -> Bool
full n x = (length x)>=n

readBound :: Bound a -> IO a
readBound (m,ne,nf,l) = lock m
	( \x -> do
	  	nx <- wait ne m empty x
		if (full l nx) 
		  then signal nf -- cause I'm about to take one out
		  else return ()
		let (v:vs) = nx
	  	return (vs,v)
	)

writeBound :: Bound a -> a -> IO ()
writeBound (m,ne,nf,l) v = lock m
	( \x -> do
		nx <- wait nf m (full l) x 
		if (empty nx) 
		  then signal ne -- cause I'm about to put one in
		  else return ()
		return ((nx++[v]),())
	)

snapBound :: Bound a -> IO [a]
snapBound (m,_,_,_) = lock m (\vs -> return (vs,vs))

isEmptyBound :: Bound a -> IO Bool
isEmptyBound b = do
	vs <- snapBound b
	return (empty vs)

isFullBound :: Bound a -> IO Bool
isFullBound b@(_,_,_,l) = do
	vs <- snapBound b
	return (full l vs)
