module Channel
(	Chan,		-- abstract

	newChan,	{- :: IO (Chan a)
				Create a FIFO buffer of unlimited size. -}	

	writeChan,	{- :: Chan a -> a -> IO ()
				Write a single value to the buffer. -}
	readChan,	{- :: Chan a -> IO a
				Read a single value from the buffer.
				NOTE: Will block if it is empty. -}

	isEmptyChan,	{- :: Chan a -> IO Bool
				Check if buffer is empty at this specific instant in time. -}

	snapChan,	{- :: Chan a -> IO [a] -- copies contents only
				Generate a list representing a snapshot view of the 
				buffer contents, the buffer is unchanged. -}

	flushChan,	{- :: Chan a -> IO [a]
				Empty the buffer. -}

	addChan,	{- :: Chan a -> [a] -> IO ()
				Write a list of values into the buffer, guaranting that
				no other value is written inbetween. -}
) where

import MutSig
import Distributed

type Chan a = (Mutex [a], Signal) 

instance Immobile (Chan a) where 
  owningPE (m,s) = owningPE m

newChan :: IO (Chan a)
newChan = do
	m <-newMutex []
	s <-newSignal
	return (m,s)

readChan :: Chan a -> IO a
readChan (m,s) = lock m
	( \x -> do
	  	(v:vs) <- wait s m null x
	  	return (vs,v)
	)

addChan :: Chan a -> [a] -> IO ()
addChan _ [] = return ()
addChan (m,s) v = lock m
	( \vs -> do
		if (null vs) 
		  then signal s
		  else return ()
		return ((vs++v),())
	)

writeChan :: Chan a -> a -> IO ()
writeChan c v = addChan c [v]

snapChan :: Chan a -> IO [a]
snapChan (m,_) = lock m (\vs -> return (vs,vs))

flushChan :: Chan a -> IO [a]
flushChan (m,_) = lock m (\vs -> return ([],vs))

isEmptyChan :: Chan a -> IO Bool
isEmptyChan c = do
	vs <- snapChan c
	return (null vs)

