/* -----------------------------------------------------------------------------
 * $Id: StgTypes.h,v 1.19 1998/06/05 10:41:25 simonm Exp $
 *
 * Various C datatypes used in the run-time system.

 * Specifically:

	StgInt8, 16, 32, 64
	StgNat8, 16, 32, 64
	StgChar, StgFloat, StgDouble

	***** All the same size: *****
	StgPtr			Basic pointer type
	StgWord			Unit of heap allocation
	StgInt			Signed version of StgWord
	StgAddr			Generic address type

 *
 * ---------------------------------------------------------------------------*/

#ifndef STGTYPES_H
#define STGTYPES_H

/*
 * First, platform-dependent definitions of size-specific integers.
 * Assume for now that the int type is 32 bits.
 * ToDo: move these into a platform-dependent file.
 */

typedef signed   char            StgInt8;
typedef unsigned char            StgNat8;

typedef signed   short           StgInt16;
typedef unsigned short           StgNat16;


#if SIZEOF_UNSIGNED_INT == 4
typedef signed   int             StgInt32;
typedef unsigned int             StgNat32;
#else
#error GHC untested on this architecture: sizeof(unisgned int) != 4
#endif

#if SIZEOF_LONG == 8
typedef signed   long            StgInt64;
typedef unsigned long            StgNat64;
#else /* assume long long is 64 bits */
typedef signed   long long int   StgInt64;
typedef unsigned long long int   StgNat64;
#endif

/*
 * Define the standard word size we'll use on this machine: make it
 * big enough to hold a pointer.
 */

#if SIZEOF_VOID_P == 8
typedef StgInt64           StgInt;
typedef StgNat64           StgWord;
#else
#if SIZEOF_VOID_P == 4
typedef StgInt32           StgInt; 
typedef StgNat32           StgWord;
#else
#error GHC untested on this architecture: sizeof(void *) != 4 or 8
#endif
#endif

typedef void*              StgAddr;

/*
 * Other commonly-used STG datatypes.
 */

typedef StgNat8            StgChar;

/*
 * If a double fits in an StgWord, don't bother using floats.
 */

#if SIZEOF_DOUBLE == SIZEOF_VOID_P
typedef double		   StgFloat;
typedef double		   StgDouble;
#define FLOATS_AS_DOUBLES  1
#else
typedef float		   StgFloat;
typedef double		   StgDouble;
#endif
                           
typedef void               StgVoid;
                           
typedef struct StgClosure_* StgClosurePtr;
typedef StgWord*           StgPtr;           /* pointer into closure           */
typedef StgWord            StgOffset;        /* byte offset within closure     */
                           
typedef StgWord            StgStablePtr;
typedef StgPtr             StgForeignPtr;

typedef StgInt             StgStackOffset;   /* offset in words! */

typedef StgWord*           StgStackPtr;

typedef StgNat8 	   StgCode;  	    /* close enough */
typedef StgCode*	   StgCodePtr;	

typedef char*		   StgByteArray;    /* the goods of a ByteArray# */

/*
  Types for the generated C functions
  take no arguments
  return a pointer to the next function to be called
  use: Ptr to Fun that returns a Ptr to Fun which returns Ptr to void
*/

typedef void  *(*(*StgFunPtr)(void))(void);
typedef StgFunPtr StgFun(void);

typedef union {
    StgWord        w;
    StgAddr        a;
    StgChar        c;
    StgFloat       f;
    StgInt         i;
    StgPtr         p;
    StgClosurePtr  cl;
    StgStackOffset offset;	/* unused? */
    StgByteArray   b;
} StgUnion;

/*
 * Shorthand forms
 */

typedef StgChar		C_;
typedef StgWord		W_;
typedef StgWord*	P_;
typedef P_*		PP_;
typedef StgInt		I_;
typedef StgAddr	        A_;
typedef const StgWord*  D_;
typedef StgFunPtr       F_;
typedef StgByteArray    B_;
typedef StgClosurePtr   L_;

/*
 * We often want to know the size of something in units of an
 * StgWord... (rounded up, of course!)
 */

#define sizeofW(t) ((sizeof(t)+sizeof(W_)-1)/sizeof(W_))

/* 
 * It's nice to be able to grep for casts
 */

#define stgCast(ty,e) ((ty)(e))

#endif STGTYPES_H

