#!/bin/bash
##############################################################################
# Time-stamp: <Wed Nov 22 2000 03:02:04 Stardate: [-30]5765.63 hwloidl>
#
# Usage: gr2ps [options] <gr-file>
#
# Create an overall activity graph from a GrAnSim (or GUM) profile.
# Transform the log file of a GrAnSim run (a .gr file) into a quasi-parallel
# profile (a .qp file) using gr2qp and then into a PostScript file using qp2ps.
# The generated PostScript file shows essentially the number of running, 
# runnable and blocked tasks during the execution of the program.
#
# Options:
#  -o <file> ... write .ps file to <file>
#  -I <str>  ... queues to be displayed (in the given order) with the encoding
#                 'a' ... active (running)
#                 'r' ... runnable
#                 'b' ... blocked
#                 'f' ... fetching
#                 'm' ... migrating
#                 's' ... sparks
#                (e.g. -I "arb" shows active, runnable, blocked tasks)
#  -i <int>  ... info level from 1 to 7; number of queues to display
#  -m        ... create mono PostScript file instead a color one.
#  -r        ... measure runtime in kcycles, megacycles rather than exactly
#  -O        ... optimise the produced .ps w.r.t. size
#		 NB: With this option info is lost. If there are several values
#		     with same x value only the first one is printed, all 
#		     others are dropped.
#  -s <str>  ... print <str> in the top right corner of the generated graph
#  -X <int>  ... forces to use <int> as runtime (x-value)
#  -Y <int>  ... forces to use <int> as max task residency (y-value)
#  -S        ... improved version of sorting events
#  -l <int>  ... length of slice in the .ps file; (default: 100)
#                small value => less memory consumption of .ps file & script
#  -d        ... Print date instead of average parallelism
#  -t        ... keep intermediate .qp file
#  -C        ... do sanity checks on log file (mainly for debugging)
#  -D        ... debugging
#  -v        ... be talkative. 
#  -h        ... print help message (this header).
#
##############################################################################

progname="`basename $0`"
args="$*"

verb=0
help=0
keep_tmp=0
mono=""
succinct_runtime=""
psfile=""
debug=""
check=""
optimise=""
info_level=""
info_mask=""
string=""
length=""
force_date=""
hack=""
hash_const=""
x_len=""
y_len=""

getopts "hvmrOHSdtX:Y:l:s:o:i:I:Q:CD" name
while [ "$name" != "?" ] ; do
  case $name in
   h) help=1;;
   v) verb=1;;
   t) keep_tmp=1;;
   m) mono="-m";;
   r) succinct_runtime="-r";;
   D) debug="-D";;
   C) check="-C";;
   O) optimise="-O";;
   d) force_date="-d";;
   H) hack="-H";;
   S) improved_sort="-S";;
   x) x_len="-X $OPTARG";;
   y) y_len="-Y $OPTARG";;
   s) string="-s $OPTARG";;
   l) length="-l $OPTARG";;
   i) info_level="-i $OPTARG";;
   I) info_mask="-I $OPTARG";;
   Q) hash_const="-Q $OPTARG";;
   o) psfile=$OPTARG;;
  esac 
  getopts "hvmrOHSdtX:Y:l:s:o:i:I:Q:CD" name
done

if [ $verb -eq 1 ]
    then verb_opt="-v "
    else verb_opt=""
fi
opts_qp="$debug $check $verb_opt $info_level $info_mask $improved_sort $hash_const "
opts_ps="$debug $check $verb_opt $optimise $mono $succinct_runtime $string $length $info_level $info_mask $force_date $x_len $y_len $hack "

shift $[ $OPTIND - 1 ]

if [ $help -eq 1 ]
 then no_of_lines=`cat $0 | awk 'BEGIN { n = 0; } \
                                 /^$/ { print n; \
                                        exit; } \
                                      { n++; }'`
      echo "`head -$no_of_lines $0`"
      exit 
fi

if [ -z "$1" ]
 then echo "Usage: $progname [options] file[.gr]"
      echo "Use -h option for details"
      exit 1;
fi

f="`basename $1 .gr`"
grfile="$f".gr
qpfile="${TMPDIR:-.}/$f".qp
ppfile="${TMPDIR:-.}/$f".pp

if [ -z "$psfile" ]
  then psfile="$f".ps
fi

if [ $verb -eq 1 ]
  then echo "Input file: $grfile"
       echo "Quasi-parallel file: $qpfile"
       echo "PP file: $ppfile"
       echo "PostScript file: $psfile"
       echo "Options for gr2qp: ${opts_qp}"
       echo "Options for qp2ps: ${opts_ps}"
       if [ -n "$mono" ]
         then echo "Producing monochrome PS file"
         else echo "Producing color PS file"
       fi
       if [ -n "$optimise" ]
         then echo "Optimisation is ON"
         else echo "Optimisation is OFF"
       fi
       if [ $keep_tmp -ne 0 ]
         then echo "Keeping intermediate file: $qpfile"
       fi
       if [ -n "$check" ]
         then echo "Consistency checks are turned ON"
         else echo "Consistency checks are turned OFF"
       fi
       if [ -n "$debug" ]
         then echo "Debugging is turned ON"
         else echo "Debugging is turned OFF"
       fi
       if [ -n "$improved_sort" ]
         then echo "Improved sort is turned ON"
         else echo "Improved sort is turned OFF"
       fi
fi


# unset noclobber
if [ ! -f "$grfile" ] 
 then
  echo "$grfile does not exist"
  exit 1
 else
  rm -f "$qpfile" "$psfile"
  prog=`head -1 "$grfile" | sed -e 's/Granularity Simulation for //'`
  echo "$prog" >| "$qpfile"
  if [ $verb -eq 1 ] 
    then echo "Executed program: $prog" 
  fi
  date >> "$qpfile"
  #date="`date`"                     # This is the date of running the script
  date="`tail +2 $grfile | head -1 | sed -e 's/Start time: //'`"
  cat "$grfile" | gr2qp ${opts_qp} >> "$qpfile"
  # Sorting is part of gr2qp now.
  #  | ghc-fool-sort | sort -n +0 -1 | ghc-unfool-sort >> "$qpfile"
  # max=`tail -2 "$qpfile" | awk '!/^Number of threads:/ { print $1; }'`
  xmax=`tail -1 "$qpfile" | awk '{ print $2; }'`
  ymax=`tail -1 "$qpfile" | awk '{ print $4; }'`
  if [ $verb -eq 1 ] 
    then echo "Total runtime: $xmax"
	 echo "Maximal number of tasks: $ymax"
  fi
  tail +3 "$qpfile" | qp2ps ${opts_ps} "$xmax" "$ymax" "$prog" "$date" >| "$psfile"
  if [ $keep_tmp -ne 1 ] 
    then rm -f "$qpfile"
    else mv "$qpfile" .
  fi
  if [ $verb -eq 1 ] 
    then echo "Scaling (maybe): ps-scale-y $psfile " 
  fi
  ps-scale-y "$psfile"
fi

  


