%
% (c) The GRASP/AQUA Project, Glasgow University, 1995-1996
%
\section[Readline]{GNU Readline Library Bindings}

This module attempts to provide a better line based editing facility
for Haskell programmers by providing access to the GNU Readline
library.  Related to this are bindings for the GNU History library
which can be found in History (at some point in the future :-).

Original version by Darren Moffat
Heavily modified in 1999 by Sven Panne <Sven.Panne@informatik.uni-muenchen.de>

NOTE: This binding is still *very* incomplete...  Volunteers?

\begin{code}
{-# OPTIONS -#include <readline/readline.h> -#include <readline/history.h> #-}

module Readline
	( KeyCode
	, RlCallbackFunction

	, rlInitialize		-- :: IO ()
	, readline		-- :: String -> IO (Maybe String)
	, addHistory		-- :: String -> IO ()
	, rlBindKey		-- :: KeyCode -> RlCallbackFunction -> IO ()
	, rlAddDefun		-- :: String  -> RlCallbackFunction -> Maybe KeyCode -> IO ()

	, rlGetReadlineName	-- :: IO String
	, rlSetReadlineName	-- :: String -> IO ()
	, rlGetLineBuffer	-- :: IO String
	, rlSetLineBuffer	-- :: String -> IO ()
	, rlGetPoint		-- :: IO Int
	, rlSetPoint		-- :: Int -> IO ()
	, rlGetEnd		-- :: IO Int
	, rlSetEnd		-- :: Int -> IO ()
	, rlGetMark		-- :: IO Int
	, rlSetMark		-- :: Int -> IO ()
	, rlSetDone		-- :: Bool -> IO ()
	, rlSetPendingInput	-- :: KeyCode -> IO ()
	, rlGetPrompt		-- :: IO String
	, rlGetTerminalName	-- :: IO String
	, rlInStream		-- :: Handle
	, rlOutStream		-- :: Handle
	) where

import Char	( ord, chr )
import Foreign	( Storable(..), Addr, nullAddr, mallocElems, free, alloca )
import IO	( Handle )
import IOExts	( IORef, newIORef, readIORef, writeIORef,
		  unsafePerformIO, freeHaskellFunctionPtr )
import Monad	( unless, zipWithM_ )
import Posix	( intToFd, fdToHandle )
import System	( getProgName )

-- SUP: Haskell has closures and I've got no clue about the return value,
--      so a better type for the callbacks is probably
--      Int {- Numeric Arg -} -> IO ()

type KeyCode = Char

type RlCallbackFunction = 
    (Int ->			-- Numeric Argument
     KeyCode ->			-- KeyCode of pressed Key
     IO Int)                    -- What's this?
\end{code}

%***************************************************************************
%*                                                                         *
\subsection[Readline-Functions]{Main Readline Functions}
%*                                                                         *
%***************************************************************************
\begin{code}

rlInitialize :: IO ()
rlInitialize = do
  n <- marshalString ""
  _casm_ ``rl_readline_name = %0;'' n
  rlSetReadlineName =<< getProgName
  b <- marshalString ""
  _casm_ ``rl_line_buffer = %0;'' b

readline :: String		-- Prompt String
	 -> IO (Maybe String)	-- Just returned line or Nothing if EOF
readline prompt =  do
   lAddr <- inString readlineAux prompt
   if lAddr == nullAddr
      then return Nothing
      else do line <- unmarshalString lAddr
              free lAddr
              return (Just line)

foreign import "readline" unsafe readlineAux :: Addr -> IO Addr

addHistory :: String		-- String to enter in history
           -> IO ()
addHistory = inString add_history

foreign import "add_history" unsafe add_history :: Addr -> IO ()

rlBindKey :: KeyCode		    -- Key to Bind to
	  -> RlCallbackFunction	    -- Function to exec on execution
	  -> IO ()
rlBindKey key cback = do
   cbAddr <- mkRlCallback (\n k -> cback n (chr k))
   ok     <- rl_bind_key (ord key) cbAddr
   if ok /= 0 then wrongKeyCode else addCbackEntry key cbAddr

foreign export dynamic mkRlCallback :: (Int -> Int -> IO Int) -> IO Addr
foreign import "rl_bind_key" rl_bind_key :: Int -> Addr -> IO Int

rlAddDefun :: String ->			-- Function Name
	      RlCallbackFunction ->	-- Function to call
	      Maybe KeyCode ->		-- Key to bind to
	      IO ()
rlAddDefun name cback mbKey = do
   cbAddr <- mkRlCallback (\n k -> cback n (chr k))
   -- ATTENTION: Memory leak due to silly readline behaviour (rl_add_defun does
   -- *not* make a copy of the function name!
   addr   <- marshalString name
   ok     <- rl_add_defun addr cbAddr (maybe (-1) ord mbKey)
   unless (ok == 0) wrongKeyCode

foreign import "rl_add_defun" unsafe rl_add_defun :: Addr -> Addr -> Int -> IO Int

-- Don't know how this should ever happen with KeyCode = Char
wrongKeyCode :: IO ()
wrongKeyCode = ioError (userError "Invalid ASCII Key Code, must be in range 0..255")

-- Global hacking for freeing callbacks

{-# notInline theCbackTable #-}
theCbackTable :: IORef [(KeyCode,Addr)]
theCbackTable = unsafePerformIO (newIORef [])

addCbackEntry :: KeyCode -> Addr -> IO ()
addCbackEntry key cbAddr = do
   cbackTable <- readIORef theCbackTable
   maybe (return ()) freeHaskellFunctionPtr (lookup key cbackTable)
   writeIORef theCbackTable
              ((key,cbAddr) : [ entry | entry@(k,_) <- cbackTable, k /= key ])
\end{code}

%***************************************************************************
%*                                                                         *
\subsection[Readline-Globals]{Global Readline Variables}
%*                                                                         *
%***************************************************************************

These are the global variables required by the readline lib. Need to
find a way of making these read/write from the Haskell side.  Should
they be in the IO Monad, should they be Mutable Variables?

\begin{code}
rlGetReadlineName :: IO String
rlGetReadlineName = unmarshalString =<< _casm_ ``%r = rl_readline_name;''

rlSetReadlineName :: String -> IO ()
rlSetReadlineName str = do
   free =<< _casm_ ``%r = rl_readline_name;''
   addr <- marshalString str
   _casm_ ``rl_readline_name = %0;'' addr

rlGetLineBuffer :: IO String
rlGetLineBuffer = unmarshalString =<< _casm_ ``%r = rl_line_buffer;''
				
rlSetLineBuffer :: String -> IO ()
rlSetLineBuffer str = do
   free =<< _casm_ ``%r = rl_line_buffer;''
   addr <- marshalString str
   _casm_ ``rl_line_buffer = %0;'' addr

rlGetPoint :: IO Int
rlGetPoint = _casm_ ``%r = rl_point;''

rlSetPoint :: Int -> IO ()
rlSetPoint point = _casm_ ``rl_point = %0;'' point
	 
rlGetEnd :: IO Int
rlGetEnd = _casm_ ``%r = rl_end;''

rlSetEnd :: Int -> IO ()
rlSetEnd end = _casm_ ``rl_end = %0;'' end

rlGetMark :: IO Int
rlGetMark = _casm_ ``%r = rl_mark;''

rlSetMark :: Int -> IO ()
rlSetMark mark = _casm_ ``rl_mark = %0;'' mark

rlSetDone :: Bool -> IO ()
rlSetDone done = _casm_ ``rl_done = %0;'' (if done then 0::Int else 1)

rlSetPendingInput :: KeyCode -> IO ()
rlSetPendingInput key = _casm_ ``rl_pending_input = %0;'' key

rlGetPrompt :: IO String
rlGetPrompt = unmarshalString =<<  _casm_ ``%r = rl_prompt;''

rlGetTerminalName :: IO String
rlGetTerminalName = unmarshalString =<< _casm_ ``%r = rl_terminal_name;''

rlInStream :: Handle
rlInStream  = unsafePerformIO (fdToHandle (intToFd ``fileno(rl_instream)''))

rlOutStream :: Handle
rlOutStream = unsafePerformIO (fdToHandle (intToFd ``fileno(rl_outstream)''))
\end{code}

%***************************************************************************
%*                                                                         *
\subsection[Readline-Util]{Miscellaneous utility functions}
%*                                                                         *
%***************************************************************************

\begin{code}
pokeString :: Addr -> String -> IO ()
pokeString buf str = do
   zipWithM_ (pokeElemOff buf) [ 0 .. ] (str ++ "\0")

inString :: (Addr -> IO a) -> String -> IO a
inString act str = alloca (length str + 1)
                          (\addr -> do pokeString addr str
                                       act addr)

marshalString :: String -> IO Addr
marshalString str = do
   let numElements = length str
   buf <- mallocElems (head str) (numElements+1)
   pokeString buf str
   return buf

unmarshalString :: Addr -> IO String
unmarshalString buf = loop 0 []
   where loop idx accu = do
            c <- peekElemOff buf idx
            if c == '\0'
               then return $ reverse accu
               else loop (idx+1) (c:accu)
\end{code}

A simple test:

main :: IO ()
main = do rlInitialize
          rlBindKey '\^X' (\nargc kc -> do print (nargc,kc); return 0)
          loop
   where loop = maybe (putStrLn "Qapla'!")
                      (\reply -> do unless (null reply) (addHistory reply)
                                    putStrLn (reply ++ "...   pItlh!")
                                    loop) =<< readline "nuqneH, ghunwI'? "
