\documentstyle[literate]{article}
\begin{document}
\section{The typechecker}

This module contains the rules used by the typechecker.



>module Typechecker where


>import Tools
>import AbsSyntax
>import Envs
>import TypeReps
>import NameSupply
>import DervTrees
>import AuxTypechecker

The type of the typechecker reflects the fact that some careful plumbing
has been done to hide unnecessary implementation details.

>typechecker :: Program -> Wombat ([TypeExp],[Dtree],TypeEnv)
>typechecker (Module name decls)  
>                = tcWherePatDecls ps ges
>                        where (ps,ges) = restructureDecls decls


At present, the typechecker only handles local pattern declarations, which
is why @tcWherePatDecls@ is called here. The declarations are passed
on in the form of a list of patterns and a list of guard, expression lists.

The function @tc@ checks expressions.

>tc :: Expr -> Wombat (TypeExp,Dtree)
>tc (Var name)  = tcVar name
>tc (Lit (LitChar name)) = tcChar name
>tc (Lit (LitString str)) = tcString str
>tc (Lit (LitNumStr nm)) = tcNum nm
>tc (Lit (LitFracStr nm)) = tcFrac nm
>tc (Con "Ptrue") = tcBool "True"
>tc (Con "Pfalse") = tcBool "False"
>tc (App e1 e2) = tcApp e1 e2					
>tc (Lambda (pat:[]) guard exp) = tcLambda pat guard exp 		
>tc (Let decls exp) = tcLet (restructureDecls decls) exp		
>tc (Where exp decls) = tcWhere (restructureDecls decls) exp	
>tc (ListComprehn exp quallist) = tcListComp exp quallist		
>tc (Case exp branches) = tcCase exp branches
>tc (Cond pred consequent alternate) = tcCond pred consequent alternate

\section{Typechecker rules and productions}

The rule implemented by @tcVar@ is

\begin{verbatim}
TAUT   A,x:s |- x:spec(A,s)
\end{verbatim}

>tcVar :: Id -> Wombat (TypeExp,Dtree)
>tcVar name
> =       (lookup name,							False)
>  `next` (\ sigtype -> (spec sigtype,					False)
>  `next` (\ tautype -> return (tautype,dtVar name sigtype tautype)))

The following functions, like @tcVar@, handle ground types.


>tcChar :: Char -> Wombat (TypeExp,Dtree)
>tcChar c = return (charTp,dtChar c)


>tcString :: String -> Wombat (TypeExp,Dtree)
>tcString str = return (stringTp,dtString str)

>tcNum :: String -> Wombat (TypeExp,Dtree)
>tcNum nm = return (numTp,dtNum nm)

>tcFrac :: String -> Wombat (TypeExp,Dtree)
>tcFrac fr = return (fracTp,dtFrac fr)

>tcBool :: String -> Wombat (TypeExp,Dtree)
>tcBool bl = return (boolTp,dtBool bl)


The rule implemented by @tcApp@ is:

\begin{verbatim}
APP    A |- f:t -> t'
       A |- a:t
       ------------------
       A |- (f a):t'
\end{verbatim}


>tcApp :: Expr -> Expr -> Wombat (TypeExp,Dtree)
>tcApp fun arg 
> =       (tc fun,							False)
>  `next` (\(ftype,fdt) -> (tc arg,					False)
>  `next` (\(atype,adt) -> (getFreshTypeVar,				False)
>  `next` (\tyvar -> (unify ftype (atype `mkArrowTp` tyvar),		False)
>  `next` (\funtype  -> return (arrowtail funtype,dtApp fdt adt)))))

The following rule is implemented by @tcLambda@:

\begin{verbatim}
ABS  A_fv(pat), B |- g:bool
     A_fv(pat), B |- e:t'
     B |= p:t
     ----------------------------------------
     A |- (lambda p | g => e): t -> t'
\end{verbatim}

>tcLambda :: Patt -> Guard -> Expr -> Wombat (TypeExp,Dtree)
>tcLambda pat guard exp
> =       (linear [pat],						False)
> `next`  (\_ -> (pattern pat,						False)
> `next`  (\ (ptype,pTypeEnv) -> (extend pTypeEnv (tc guard),		False)
> `next`  (\ (gtype,gdt) -> (unify gtype boolTp,			False)
> `next`  (\ _ -> (extend pTypeEnv (subHiddenAsgn (tc exp)),		False)
> `next`  (\ (etype,edt) -> (substitute ptype,				False)
> `next`  (\ newPtype  -> return (newPtype `mkArrowTp` etype,
>                                 dtLambda pat ptype gdt edt)))))))

The rule implemented by @tcLet@ is

\begin{verbatim}
LET       A_fv(p1),...,fv(pi) |- e11:t1
              ...
          A_fv(p1),...,fv(pi) |- e1n:t1
          A_fv(p1),...,fv(pi) |- e21:t2
              ...
          A_fv(p1),...,fv(pi) |- e2m:t2
          ...
          A_fv(p1),...,fv(pi) |- ei1:ti
              ...
          A_fv(p1),...,fv(pi) |- eik:ti
          A_fv(p1),...,fv(pi) |- g11:bool
              ...
          A_fv(p1),...,fv(pi) |- g1n:bool
          A_fv(p1),...,fv(pi) |- g21:bool
              ...
          A_fv(p1),...,fv(pi) |- g2m:bool
          ...
          A_fv(p1),...,fv(pi) |- gi1:bool
              ...
          A_fv(p1),...,fv(pi) |- gik:bool
          B1 |= p1:t1
          ...
          Bi |= pi:ti
          A_fv(p1),...,fv(pi), GEN(A,B1),...,GEN(A,Bi) |- e':t'
          ----------------------------------------------------
          A |- let    p1 | g11 = e11
                         ...
                         | g1n = e1n
                      p2 | g21 = e21
                         ...
                         | g2m = e2m
                     ...
                      pi | gi1 = ei1
                         ...
                         | gik = eik
                 in e': t'

     singleGen (A,t) = forall a1,...,an.t  where ai not free in A

     gen (A,empty) = empty
     gen (A,B U {x:t}) = gen (A,B) U {x:singleGen(A,t)}
\end{verbatim}

Let will disappear when dependency analysis is integrated.


>tcLet :: ([Patt],[[(Guard,Expr)]]) -> Expr -> Wombat (TypeExp,Dtree)
>tcLet (ps,ges) e
> =       (linear ps, 							False)
>  `next` (\ _ -> (tcLetPatDecls ps ges,				False)
>  `next` (\ (ts,dsts,penv) -> (gen penv,				False)
>  `next` (\ penv' -> (subHiddenAsgn (extend penv' (tc e)),		False)
>  `next` (\ (et,edt) -> return (et,dtLet dsts edt)))))


Pattern declarations in let expressions are handled by @tcLetPatDecls@,
which first calls @tcLetPatDeclsPairlsts@ to check the right hand sides,
after which it unifies these with the pattern types.

>tcLetPatDecls :: [Patt] -> [[(Guard,Expr)]] 
>                   -> Wombat ([TypeExp],[Dtree],TypeEnv)
>tcLetPatDecls ps ges
> =       (patterns ps,							False)
>  `next` (\ (ptypes,penvs) -> (tcLetPatDeclsPairlsts ps ges,		False)
>  `next` (\ (etypes,dsts) -> (unifyLst (ptypes `zip` etypes),		False)
>  `next` (\ etypes' -> return (etypes',
>                               dtLetPatDecls dsts,
>                               concatTypeEnvs penvs))))

This function calls @tcLetPatDeclPairlst@ to get the type of a pattern
definition, then adds that type and derivation tree, created while
checking the guard,expression pairs, to the rest of the result lists.

>tcLetPatDeclsPairlsts :: [Patt] -> [[(Guard,Expr)]]
>                                               -> Wombat ([TypeExp],[Dtree])
>tcLetPatDeclsPairlsts [] [] = return ([],[])
>tcLetPatDeclsPairlsts (p:ps) (gelst:gelsts) 
> =       (tcLetPatDeclPairlst p gelst,					False)
>  `next` (\ (t,dt) -> (tcLetPatDeclsPairlsts ps gelsts,		False)
>  `next` (\ (ts,dts) -> (substitute t,					False)
>  `next` (\ t' ->  return (t':ts,dtLetPatDeclsPairlsts dt dts))))

This function accumulates a type and derivation tree for the guard
expression pairs of a pattern declaration, by calling
@tcLetPatDeclsPairTypes@. It then unifies all of the returned types,
returning a single type, and inserts the derivation trees into a tree
for the whole definition.

>tcLetPatDeclPairlst :: Patt -> [(Guard,Expr)] -> Wombat (TypeExp,Dtree)
>tcLetPatDeclPairlst p pairlst
> =       (tcLetPatDeclPairTypes pairlst,				False)
>  `next` (\ (ts,dts) -> (unifyExps ts,					False)
>  `next` (\ t -> return (t,dtLetPatDeclPairlst p dts)))

Finally, this is the function that checks a single guard expression
pair in a pattern declaration list.

>tcLetPatDeclPairTypes :: [(Guard,Expr)] -> Wombat ([TypeExp],[(Dtree,Dtree)])
>tcLetPatDeclPairTypes [] = return ([],[])
>tcLetPatDeclPairTypes ((g,e):ges)
> =       (subHiddenAsgn (tc g),					False)
>  `next` (\ (gt,gdt) -> (unify gt boolTp,				False)
>  `next` (\_ -> (subHiddenAsgn (tc e),					False)
>  `next` (\ (et,edt) -> (tcLetPatDeclPairTypes ges,			False)
>  `next` (\ (ts,dts) -> (substitute et,				False)
>  `next` (\ et' -> return (et':ts,dtLetPatDeclPairTypes (gdt,edt) dts))))))


The rule implemented by @tcWhere@ is

\begin{verbatim}
REC       A_fv(p1),...,fv(pi),B1,...,Bi |- e11:t1
              ...
          A_fv(p1),...,fv(pi),B1,...,Bi |- e1n:t1
          A_fv(p1),...,fv(pi),B1,...,Bi |- e21:t2
              ...
          A_fv(p1),...,fv(pi),B1,...,Bi |- e2m:t2
          ...
          A_fv(p1),...,fv(pi),B1,...,Bi |- ei1:ti
              ...
          A_fv(p1),...,fv(pi),B1,...,Bi |- eik:ti
          A_fv(p1),...,fv(pi),B1,...,Bi |- g11:bool
              ...
          A_fv(p1),...,fv(pi),B1,...,Bi |- g1n:bool
          A_fv(p1),...,fv(pi),B1,...,Bi |- g21:bool
              ...
          A_fv(p1),...,fv(pi),B1,...,Bi |- g2m:bool
          ...
          A_fv(p1),...,fv(pi),B1,...,Bi |- gi1:bool
              ...
          A_fv(p1),...,fv(pi),B1,...,Bi |- gik:bool
          B1 |= p1:t1
          ...
          Bi |= pi:ti
          A_fv(p1),...,fv(pi), GEN(A,B1),...,GEN(A,Bi) |- e':t'
          ----------------------------------------------------
          A |- rec    p1 | g11 = e11
                         ...
                         | g1n = e1n
                      p2 | g21 = e21
                         ...
                         | g2m = e2m
                     ...
                      pi | gi1 = ei1
                         ...
                         | gik = eik
\end{verbatim}


>tcWhere :: ([Patt],[[(Guard,Expr)]]) -> Expr -> Wombat (TypeExp,Dtree)
>tcWhere (ps,ges) e
> =       (linear ps,							False)
>  `next` (\_ -> (tcWherePatDecls ps ges,				False)
>  `next` (\ (ts,dsts,penv) -> (gen penv,				False)
>  `next` (\ penv' -> (extend penv' (subHiddenAsgn (tc e)),		False)
>  `next` (\ (et,edt) -> return (et,dtWhere dsts edt)))))

Pattern declarations in let expressions are handled by @tcWherePatDecls@,
which first calls @tcWherePatDeclsPairlsts@ to check the right hand sides,
after which it unifies these with the pattern types. The pattern types
need to be constrained by whatever was learned by checking the right hand
sides, so the current substitution is applied to them before the unification.

>tcWherePatDecls :: [Patt] -> [[(Guard,Expr)]] 
>                              -> Wombat ([TypeExp],[Dtree],TypeEnv)
>tcWherePatDecls ps ges
> =       (patterns ps,							False)
>  `next` (\ (ptypes,penvs) -> (tcWherePatDeclsPairlsts ps penvs ges,	False)
>  `next` (\ (etypes,dsts) -> (substituteLst ptypes,			False)
>  `next` (\ ptypes' -> (unifyLst (ptypes' `zip` etypes),		False)
>  `next` (\ etypes' -> return (etypes',
>                               dtWherePatDecls dsts,
>                               concatTypeEnvs penvs)))))

This function calls @tcWherePatDeclPairlst@ to get the type of a pattern
definition, then adds that type and derivation tree, created while
checking the guard,expression pairs, to the rest of the result lists.

>tcWherePatDeclsPairlsts :: [Patt] -> [TypeEnv] -> [[(Guard,Expr)]]
>                                    -> Wombat ([TypeExp],[Dtree])
>tcWherePatDeclsPairlsts [] [] [] = return ([],[])
>tcWherePatDeclsPairlsts (p:ps) (penv:penvs) (gelst:gelsts) 
> =      (tcWherePatDeclPairlst p penv gelst,				False)
> `next` (\ (t,dt) -> (tcWherePatDeclsPairlsts ps penvs gelsts,		False)
> `next` (\ (ts,dts) -> (substitute t,					False)
> `next` (\ t' -> return (t':ts,dtWherePatDeclsPairlsts dt dts))))

This function accumulates a type and derivation tree for the guard
expression pairs of a pattern declaration, by calling
@tcWherePatDeclsPairTypes@. It then unifies all of the returned types,
returning a single type, and inserts the derivation trees into a tree
for the whole definition.


>tcWherePatDeclPairlst :: Patt -> TypeEnv -> [(Guard,Expr)]
>                                               -> Wombat (TypeExp,Dtree)
>tcWherePatDeclPairlst p penv pairlst
> =       (tcWherePatDeclPairTypes penv pairlst,			False)
>  `next` (\ (ts,dts) -> (unifyExps ts,					False)
>  `next` (\ t -> return (t,dtWherePatDeclPairlst p dts)))

Finally, this is the function that checks a single guard expression
pair in a pattern declaration list.

>tcWherePatDeclPairTypes :: TypeEnv -> [(Guard,Expr)] 
>                             -> Wombat ([TypeExp],[(Dtree,Dtree)])
>tcWherePatDeclPairTypes penv [] = return ([],[])
>tcWherePatDeclPairTypes penv ((g,e):ges)
> =       (subHiddenAsgn (extend penv (tc g)),				False)
>  `next` (\ (gt,gdt) -> (unify gt boolTp,				False)
>  `next` (\_ -> (subHiddenAsgn (extend penv (tc e)),			False)
>  `next` (\ (et,edt) -> (tcWherePatDeclPairTypes penv ges,		False)
>  `next` (\ (ts,dts) -> (substitute et,				False)
>  `next` (\ et' -> return (et':ts,
>			    dtWherePatDeclPairTypes (gdt,edt) dts))))))


Rules for list comprehensions:


\begin{verbatim}
    A |- qs => B
    A,B |- e:t
    -------------------
    A |- [e | qs] : [t]


    A |- q => B'
    A,B' |- p => B''
    --------------------
    A |- [q;p] => B' B''


    B |= p:t
    A |- e:[t]
    ----------------
    A |- p <- e => B


    A |- b:bool
    ------------
    A |- b => {}
\end{verbatim}


>tcListComp :: Expr -> [Qual] -> Wombat (TypeExp,Dtree)
>tcListComp e quallist
> =       (tcQuals quallist,						False)
>  `next` (\ (penv,dts) -> (extend penv (subHiddenAsgn (tc e)),		False)
>  `next` (\ (et,edt) -> return (et,dtListComp dts edt)))


>tcQuals :: [Qual] -> Wombat (TypeEnv,[Dtree])
>tcQuals [] = return ([],[])
>tcQuals (q:qlst)
> =       (tcQual q,							False)
>  `next` (\ (penv,dt) -> (tcQuals qlst,				False)
>  `next` (\ (penvs,dts') -> (subAsgn penv,				False)
>  `next` (\ penv' -> return (penv'++penvs,dtQuals dt dts'))))

>tcQual :: Qual -> Wombat (TypeEnv,Dtree)
>tcQual (BindQual pat exp)
> =       (pattern pat,							False)
>  `next` (\ (ptype,penv) -> (tc exp,					False)
>  `next` (\ (et,dt) -> (unify et (mkListTp ptype),			False)
>  `next` (\ et' -> return (penv,dtBindQual pat dt))))


>tcQual (BoolQual bl)
> =       (tc bl,							False)
>  `next` (\ (bt,bdt) -> (unify bt boolTp,				False)
>  `next` (\ newbt -> return ([],dtBoolQual bdt)))

Rules for Case expressions:

\begin{verbatim}
CASE  
      A |- exp:t
      B1 |= p1:t
      ...
      Bn |= pn:t
      A_fv(p1), B1 |- g1:bool
      ...
      A_fv(pn), Bn |- gn:bool
      A_fv(p1), B1 |- e1:t'
      ...
      A_fv(pn), Bn |- en:t'
      ------------------------------------------------------  

      A |- case exp of {p1 | g1 -> e1,...,pn | gn -> en}: t'
\end{verbatim}

First @exp@ is checked, then the branches. The types returned are
unified, and one of them returned.

>tcCase :: Expr -> [(Patt,Guard,Expr)] -> Wombat (TypeExp,Dtree)
>tcCase exp branches
> =       (tc exp,							False)
>  `next` (\ (etype,edt) -> (tcCaseBranches etype branches,		False)
>  `next` (\ (btypes,bdts) -> (unifyExps btypes,			False)
>  `next` (\ t -> return (t,dtCase edt bdts))))

>tcCaseBranches :: TypeExp -> [(Patt,Guard,Expr)] -> Wombat ([TypeExp],[Dtree])
>tcCaseBranches et [] = return ([],[])
>tcCaseBranches et (b:bs)
> =       (tcCaseBranch et b,						False)
>  `next` (\ (bt,bdt) -> (tcCaseBranches et bs,				False)
>  `next` (\ (ts,dts) -> (substitute bt,				False)
>  `next` (\ bt' -> return (bt':ts,dtCaseBranches bdt dts))))

>tcCaseBranch :: TypeExp -> (Patt,Guard,Expr) -> Wombat (TypeExp,Dtree)
>tcCaseBranch et (pat,guard,exp)
> =       (pattern pat,							False)
>  `next` (\ (ptype,penv) -> (unify ptype et,				False)
>  `next` (\ et' ->  (tc guard,						False)
>  `next` (\ (gt,gdt) -> (unify gt boolTp,				False)
>  `next` (\_ ->  (subHiddenAsgn (tc exp),				False)
>  `next` (\ (et,edt) -> return (et,dtCaseBranch pat gdt edt))))))

The rule for conditionals is:

\begin{verbatim}
  A |- p:bool
  A |- b1:t'
  A |- b2:t'
  ------------------------------
  A |- (if p then b1 else b2):t'
\end{verbatim}

>tcCond :: Expr -> Expr -> Expr -> Wombat (TypeExp,Dtree)
>tcCond pred b1 b2
> =       (tc pred,							False)
>  `next` (\ (predt,pdt) -> (unify predt boolTp,			False)
>  `next` (\ _ -> (subHiddenAsgn (tc b1),				False)
>  `next` (\ (b1t,b1dt) -> (subHiddenAsgn (tc b2),			False)
>  `next` (\ (b2t,b2dt) -> (unifyExps [b1t,b2t],			False)
>  `next` (\ condt -> return (condt,dtCond pdt b1dt b2dt))))))

\printindex
\end{document}
